// Main Application
class GameRewardsApp {
    constructor() {
        this.tg = window.Telegram.WebApp;
        this.userData = {
            id: null,
            name: '',
            username: '',
            balance: 0,
            adsWatched: 0,
            adsRequired: 5,
            referralCode: '',
            referredCount: 0,
            rewardPerAd: 0.10
        };
        
        this.init();
    }
    
    async init() {
        // Expand WebApp to full screen
        this.tg.expand();
        
        // Get Telegram user data
        const tgUser = this.tg.initDataUnsafe.user;
        this.userData.id = tgUser.id;
        this.userData.name = tgUser.first_name || tgUser.username || 'User';
        this.userData.username = tgUser.username ? `@${tgUser.username}` : '';
        
        // Show user avatar initial
        this.showUserAvatar();
        
        // Load user data from backend
        await this.loadUserData();
        
        // Initialize UI
        this.initUI();
        
        // Load Monetag script
        this.loadMonetagScript();
        
        // Hide loading screen, show main content
        document.getElementById('loading-screen').style.display = 'none';
        document.getElementById('main-content').style.display = 'block';
    }
    
    showUserAvatar() {
        const avatarEl = document.getElementById('user-avatar');
        if (this.userData.name) {
            avatarEl.textContent = this.userData.name.charAt(0).toUpperCase();
        }
    }
    
    async loadUserData() {
        try {
            const response = await this.apiRequest('getUserData', {
                userId: this.userData.id
            });
            
            if (response.success) {
                this.userData.balance = parseFloat(response.balance);
                this.userData.adsWatched = parseInt(response.adsWatched);
                this.userData.adsRequired = parseInt(response.adsRequired);
                this.userData.rewardPerAd = parseFloat(response.rewardPerAd);
                this.userData.referralCode = response.referralCode;
                this.userData.referredCount = parseInt(response.referredCount);
                
                // Update reward display
                document.getElementById('watch-ad-reward').textContent = 
                    `+${this.userData.rewardPerAd.toFixed(2)}`;
            }
        } catch (error) {
            console.error('Error loading user data:', error);
            this.tg.showAlert('Failed to load your data. Please try again.');
        }
    }
    
    initUI() {
        // User profile
        document.getElementById('user-name').textContent = this.userData.name;
        document.getElementById('user-username').textContent = this.userData.username;
        document.getElementById('user-balance').textContent = this.userData.balance.toFixed(2);
        document.getElementById('ads-watched').textContent = this.userData.adsWatched;
        
        // Ads remaining
        const remainingAds = Math.max(0, this.userData.adsRequired - this.userData.adsWatched);
        document.getElementById('ads-remaining').textContent = remainingAds;
        
        // Games section
        if (remainingAds > 0) {
            document.getElementById('games-locked').style.display = 'block';
            document.getElementById('games-container').style.display = 'none';
        } else {
            document.getElementById('games-locked').style.display = 'none';
            document.getElementById('games-container').style.display = 'block';
        }
        
        // Referral link
        if (this.userData.referralCode) {
            const referralLink = `https://t.me/your_bot?start=${this.userData.referralCode}`;
            document.getElementById('referral-link').value = referralLink;
            document.getElementById('referral-count').textContent = this.userData.referredCount;
            
            // Copy referral link
            document.getElementById('copy-referral-btn').addEventListener('click', () => {
                navigator.clipboard.writeText(referralLink).then(() => {
                    this.tg.showAlert('Referral link copied to clipboard!');
                });
            });
        }
        
        // Withdraw button
        document.getElementById('withdraw-btn').addEventListener('click', this.handleWithdraw.bind(this));
        document.getElementById('withdraw-amount').addEventListener('input', this.validateWithdraw.bind(this));
        
        // Watch ad button
        document.getElementById('watch-ad-btn').addEventListener('click', this.startAdWatch.bind(this));
    }
    
    validateWithdraw() {
        const amount = parseFloat(document.getElementById('withdraw-amount').value);
        const wallet = document.getElementById('wallet-address').value.trim();
        const btn = document.getElementById('withdraw-btn');
        
        btn.disabled = !(amount >= 10 && wallet.length > 0 && amount <= this.userData.balance);
    }
    
    async handleWithdraw() {
        const amount = parseFloat(document.getElementById('withdraw-amount').value);
        const wallet = document.getElementById('wallet-address').value.trim();
        
        if (amount < 10) {
            this.tg.showAlert('Minimum withdrawal amount is 10 coins');
            return;
        }
        
        if (amount > this.userData.balance) {
            this.tg.showAlert('Insufficient balance');
            return;
        }
        
        if (!wallet) {
            this.tg.showAlert('Please enter your wallet address');
            return;
        }
        
        try {
            const response = await this.apiRequest('requestWithdraw', {
                userId: this.userData.id,
                amount: amount,
                wallet: wallet
            });
            
            if (response.success) {
                this.userData.balance -= amount;
                document.getElementById('user-balance').textContent = this.userData.balance.toFixed(2);
                this.tg.showAlert(`Withdrawal request for ${amount} coins submitted successfully!`);
                
                // Clear form
                document.getElementById('withdraw-amount').value = '';
                document.getElementById('wallet-address').value = '';
                document.getElementById('withdraw-btn').disabled = true;
            } else {
                this.tg.showAlert(response.message || 'Withdrawal failed. Please try again.');
            }
        } catch (error) {
            console.error('Withdrawal error:', error);
            this.tg.showAlert('An error occurred. Please try again.');
        }
    }
    
    async startAdWatch() {
        const btn = document.getElementById('watch-ad-btn');
        btn.disabled = true;
        document.getElementById('watch-ad-text').textContent = 'Loading Ad...';
        
        try {
            // Initialize ad watch
            const initResponse = await this.apiRequest('initAdWatch', {
                userId: this.userData.id
            });
            
            if (!initResponse.success) {
                throw new Error(initResponse.message || 'Failed to initialize ad');
            }
            
            // Show ad container and timer
            document.getElementById('ad-timer').style.display = 'block';
            document.getElementById('watch-ad-text').textContent = 'Watching Ad...';
            
            // Start countdown
            const duration = 30; // 30 seconds
            let secondsLeft = duration;
            document.getElementById('countdown').textContent = secondsLeft;
            
            const timerInterval = setInterval(() => {
                secondsLeft--;
                document.getElementById('countdown').textContent = secondsLeft;
                
                // Update progress bar
                const progressPercent = ((duration - secondsLeft) / duration) * 100;
                document.getElementById('timer-progress').style.width = `${progressPercent}%`;
                
                if (secondsLeft <= 0) {
                    clearInterval(timerInterval);
                    this.completeAdWatch();
                }
            }, 1000);
            
        } catch (error) {
            console.error('Ad watch error:', error);
            this.tg.showAlert(error.message || 'Failed to start ad. Please try again.');
            btn.disabled = false;
            document.getElementById('watch-ad-text').textContent = 'Watch Ad & Earn Coins';
        }
    }
    
    async completeAdWatch() {
        try {
            const response = await this.apiRequest('completeAdWatch', {
                userId: this.userData.id
            });
            
            if (response.success) {
                // Update user data
                this.userData.balance += this.userData.rewardPerAd;
                this.userData.adsWatched++;
                
                // Update UI
                document.getElementById('user-balance').textContent = this.userData.balance.toFixed(2);
                document.getElementById('ads-watched').textContent = this.userData.adsWatched;
                
                // Check if games should be unlocked
                const remainingAds = Math.max(0, this.userData.adsRequired - this.userData.adsWatched);
                document.getElementById('ads-remaining').textContent = remainingAds;
                
                if (remainingAds === 0) {
                    document.getElementById('games-locked').style.display = 'none';
                    document.getElementById('games-container').style.display = 'block';
                }
                
                // Show success message
                this.tg.showAlert(`You earned ${this.userData.rewardPerAd.toFixed(2)} coins!`);
            } else {
                throw new Error(response.message || 'Failed to complete ad watch');
            }
        } catch (error) {
            console.error('Complete ad error:', error);
            this.tg.showAlert(error.message || 'Failed to complete ad. Please try again.');
        } finally {
            // Reset ad UI
            document.getElementById('ad-timer').style.display = 'none';
            document.getElementById('watch-ad-btn').disabled = false;
            document.getElementById('watch-ad-text').textContent = 'Watch Ad & Earn Coins';
        }
    }
    
    async loadMonetagScript() {
        try {
            const response = await this.apiRequest('getMonetagScript');
            
            if (response.success && response.script) {
                const script = document.createElement('script');
                script.textContent = response.script;
                document.getElementById('ad-container').appendChild(script);
            }
        } catch (error) {
            console.error('Failed to load Monetag script:', error);
        }
    }
    
    async apiRequest(action, data = {}) {
        const requestData = {
            action: action,
            ...data
        };
        
        const response = await fetch('https://rxhoster.top/api.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(requestData)
        });
        
        return await response.json();
    }
}

// Initialize the app when Telegram WebApp is ready
Telegram.WebApp.ready();
new GameRewardsApp();