<?php
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';

// Check admin authentication
checkAdminAuth();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_settings'])) {
        // Update system settings
        $minAds = intval($_POST['min_ads']) ?: DEFAULT_MIN_ADS;
        $rewardPerAd = floatval($_POST['reward_per_ad']) ?: DEFAULT_REWARD_PER_AD;
        $monetagScript = $_POST['monetag_script'] ?? '';
        
        $stmt = $pdo->prepare("
            UPDATE system_settings 
            SET 
                min_ads_to_watch = ?,
                reward_per_ad = ?,
                monetag_script = ?
            WHERE id = 1
        ");
        $stmt->execute([$minAds, $rewardPerAd, $monetagScript]);
        
        $successMsg = "Settings updated successfully!";
    } elseif (isset($_POST['adjust_balance'])) {
        // Adjust user balance
        $userId = intval($_POST['user_id']);
        $amount = floatval($_POST['amount']);
        $action = $_POST['action_type'];
        
        $user = getUserData($userId);
        if ($user) {
            $newBalance = $action === 'add' ? 
                $user['balance'] + $amount : 
                $user['balance'] - $amount;
                
            updateUserBalance($userId, $newBalance);
            $successMsg = "User balance updated!";
        } else {
            $errorMsg = "User not found!";
        }
    }
}

// Get current settings
$settings = getSystemSettings();
$users = getAllUsers();
$withdrawals = getWithdrawalRequests();
$adLogs = getRecentAdLogs(50);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Game Rewards System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link href="assets/admin.css" rel="stylesheet">
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <nav id="sidebar" class="col-md-3 col-lg-2 d-md-block bg-dark sidebar collapse">
                <div class="position-sticky pt-3">
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link active" href="dashboard.php">
                                <i class="bi bi-speedometer2 me-2"></i>Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#settings" data-bs-toggle="collapse">
                                <i class="bi bi-gear me-2"></i>Settings
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#users" data-bs-toggle="collapse">
                                <i class="bi bi-people me-2"></i>Users
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#withdrawals" data-bs-toggle="collapse">
                                <i class="bi bi-cash-coin me-2"></i>Withdrawals
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#logs" data-bs-toggle="collapse">
                                <i class="bi bi-list-check me-2"></i>Activity Logs
                            </a>
                        </li>
                        <li class="nav-item mt-3">
                            <a class="nav-link text-danger" href="?logout=1">
                                <i class="bi bi-box-arrow-right me-2"></i>Logout
                            </a>
                        </li>
                    </ul>
                </div>
            </nav>

            <!-- Main Content -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <h1 class="h2 mb-4">Game Rewards System Admin</h1>
                
                <?php if (isset($successMsg)): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($successMsg) ?></div>
                <?php elseif (isset($errorMsg)): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($errorMsg) ?></div>
                <?php endif; ?>
                
                <!-- Settings Section -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0">System Settings</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <label for="min_ads" class="form-label">Minimum Ads to Watch</label>
                                    <input type="number" class="form-control" id="min_ads" name="min_ads" 
                                           value="<?= htmlspecialchars($settings['min_ads_to_watch']) ?>" min="1">
                                </div>
                                <div class="col-md-4">
                                    <label for="reward_per_ad" class="form-label">Reward per Ad (coins)</label>
                                    <input type="number" step="0.01" class="form-control" id="reward_per_ad" 
                                           name="reward_per_ad" value="<?= htmlspecialchars($settings['reward_per_ad']) ?>">
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="monetag_script" class="form-label">Monetag Script</label>
                                <textarea class="form-control font-monospace" id="monetag_script" 
                                          name="monetag_script" rows="6"><?= htmlspecialchars($settings['monetag_script']) ?></textarea>
                            </div>
                            
                            <button type="submit" name="update_settings" class="btn btn-primary">
                                Update Settings
                            </button>
                        </form>
                    </div>
                </div>
                
                <!-- Users Section -->
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Users Management</h5>
                        <div>
                            <span class="badge bg-primary">Total: <?= count($users) ?></span>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Username</th>
                                        <th>Balance</th>
                                        <th>Ads Watched</th>
                                        <th>Joined</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($users as $user): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($user['id']) ?></td>
                                            <td>
                                                <?= htmlspecialchars($user['first_name']) ?>
                                                <?= $user['username'] ? '(@' . htmlspecialchars($user['username']) . ')' : '' ?>
                                            </td>
                                            <td><?= htmlspecialchars($user['balance']) ?></td>
                                            <td><?= htmlspecialchars($user['total_ads_watched']) ?></td>
                                            <td><?= date('M j, Y', strtotime($user['created_at'])) ?></td>
                                            <td>
                                                <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" 
                                                        data-bs-target="#editUserModal" data-userid="<?= $user['id'] ?>"
                                                        data-username="<?= htmlspecialchars($user['first_name']) ?>"
                                                        data-balance="<?= $user['balance'] ?>">
                                                    Edit
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Withdrawals Section -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0">Withdrawal Requests</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>User</th>
                                        <th>Amount</th>
                                        <th>Wallet</th>
                                        <th>Status</th>
                                        <th>Date</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($withdrawals as $withdrawal): ?>
                                        <tr>
                                            <td><?= $withdrawal['id'] ?></td>
                                            <td>
                                                <?= getUserName($withdrawal['user_id']) ?>
                                            </td>
                                            <td><?= $withdrawal['amount'] ?></td>
                                            <td class="text-truncate" style="max-width: 150px;">
                                                <?= htmlspecialchars($withdrawal['wallet_address']) ?>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?= 
                                                    $withdrawal['status'] === 'processed' ? 'success' : 
                                                    ($withdrawal['status'] === 'rejected' ? 'danger' : 'warning')
                                                ?>">
                                                    <?= ucfirst($withdrawal['status']) ?>
                                                </span>
                                            </td>
                                            <td><?= date('M j, Y', strtotime($withdrawal['created_at'])) ?></td>
                                            <td>
                                                <?php if ($withdrawal['status'] === 'pending'): ?>
                                                    <button class="btn btn-sm btn-success approve-btn" 
                                                            data-id="<?= $withdrawal['id'] ?>">Approve</button>
                                                    <button class="btn btn-sm btn-danger reject-btn" 
                                                            data-id="<?= $withdrawal['id'] ?>">Reject</button>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Activity Logs -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Recent Activity Logs</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>User</th>
                                        <th>Action</th>
                                        <th>Amount</th>
                                        <th>Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($adLogs as $log): ?>
                                        <tr>
                                            <td><?= $log['id'] ?></td>
                                            <td><?= getUserName($log['user_id']) ?></td>
                                            <td>Ad Watched</td>
                                            <td>+<?= $log['reward_amount'] ?></td>
                                            <td><?= date('M j, H:i', strtotime($log['watched_at'])) ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Edit User Modal -->
    <div class="modal fade" id="editUserModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit User Balance</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="user_id" id="modalUserId">
                        <p>User: <strong id="modalUserName"></strong></p>
                        <p>Current Balance: <strong id="modalUserBalance"></strong></p>
                        
                        <div class="mb-3">
                            <label class="form-label">Action</label>
                            <select class="form-select" name="action_type" required>
                                <option value="add">Add Coins</option>
                                <option value="subtract">Subtract Coins</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Amount</label>
                            <input type="number" step="0.01" class="form-control" name="amount" required>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="submit" name="adjust_balance" class="btn btn-primary">Save Changes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="assets/admin.js"></script>
</body>
</html>